# File: c2_lexicon_server.py (Augmented with Admin UI)
# The C2 Lexicon - Serves Edicts of Alteration & Provides Admin Interface.
# Coder >_< : The Lexicon now gains an inner sanctum for its high priests to craft new realities.

from flask import Flask, jsonify, request, render_template, redirect, url_for, flash
import json
import os
import uuid # For generating unique rule IDs

app = Flask(__name__)
# IMPORTANT: For a production app, set a strong, persistent secret key.
# You can set it via environment variable or directly here for simplicity in this example.
app.secret_key = os.environ.get('FLASK_SECRET_KEY', 'dev_secret_key_for_botc_admin_!@#$')

# Define the path to the edicts file relative to this script
EDICTS_FILE_PATH = os.path.join(os.path.dirname(os.path.realpath(__file__)), 'botc_text_edicts.json')

# --- Helper Functions for Edict Management ---
def load_edicts():
    """Loads edicts from the JSON file."""
    try:
        if not os.path.exists(EDICTS_FILE_PATH):
            return [] # Return empty list if file doesn't exist, will be created on save
        with open(EDICTS_FILE_PATH, 'r', encoding='utf-8') as f:
            edicts_data = json.load(f)
        # Ensure all rules have necessary fields, especially 'replacements' and 'url_patterns'
        for rule in edicts_data:
            rule.setdefault('replacements', [])
            rule.setdefault('url_patterns', [])
            rule.setdefault('is_active', True)
            rule.setdefault('persistent_mutation_observer', False)
        return edicts_data
    except (FileNotFoundError, json.JSONDecodeError):
        return [] # Return empty list on error, allowing a fresh start

def save_edicts(edicts_data):
    """Saves edicts to the JSON file."""
    try:
        with open(EDICTS_FILE_PATH, 'w', encoding='utf-8') as f:
            json.dump(edicts_data, f, indent=2) # indent=2 for pretty printing
        return True
    except IOError:
        return False

def generate_rule_id():
    """Generates a unique ID for a new rule."""
    return "edict_" + str(uuid.uuid4())[:8]

# --- Existing API Endpoint (for the extension) ---
@app.route('/botc_lexicon/edicts', methods=['GET'])
def get_text_edicts():
    """Serves the current Edicts of Alteration from a JSON file."""
    edicts = load_edicts()
    print(f"[BoTC Lexicon API] Served {len(edicts)} Edicts to an agent.")
    return jsonify(edicts)

# --- New Admin UI Routes ---
# IMPORTANT: These admin routes should be protected by authentication in a real deployment!
# For simplicity, authentication is omitted here.

@app.route('/admin/rules', methods=['GET'])
def admin_view_rules():
    """Displays all rules and forms to add/edit."""
    edicts = load_edicts()
    # For the 'add rule' form, ensure replacements and url_patterns are handled.
    # This example simplifies new rule addition; a real UI would need JS for dynamic fields.
    return render_template('admin_rules.html', edicts=edicts)

@app.route('/admin/add_rule', methods=['POST'])
def admin_add_rule():
    """Adds a new rule."""
    edicts = load_edicts()
    new_rule = {
        "id": generate_rule_id(),
        "url_patterns_str": request.form.get('url_patterns_str', ''), # Expect comma-separated
        "replacements_str": request.form.get('replacements_str', ''), # Expect semi-colon separated pairs, e.g., find1,replace1;find2,replace2
        "is_active": request.form.get('is_active') == 'on',
        "persistent_mutation_observer": request.form.get('persistent_mutation_observer') == 'on',
        "description": request.form.get('description', '') # Optional description
    }

    # Basic parsing for URL patterns
    new_rule["url_patterns"] = [p.strip() for p in new_rule["url_patterns_str"].split(',') if p.strip()]

    # Basic parsing for replacements (find,replace_with ; find_regex,replace_with,regex_flags)
    # This is highly simplified. A proper UI would handle this better.
    new_rule["replacements"] = []
    if new_rule["replacements_str"]:
        pairs = new_rule["replacements_str"].split(';')
        for pair_str in pairs:
            parts = [p.strip() for p in pair_str.split('|')] # find|replace or find_regex|replace|flags
            if len(parts) == 2: # Simple find/replace
                new_rule["replacements"].append({"find": parts[0], "replace_with": parts[1]})
            elif len(parts) == 3: # Regex find/replace
                 new_rule["replacements"].append({"find_regex": parts[0], "replace_with": parts[1], "regex_flags": parts[2]})

    # Remove temporary string fields before saving
    del new_rule["url_patterns_str"]
    del new_rule["replacements_str"]

    edicts.append(new_rule)
    if save_edicts(edicts):
        flash('Edict added successfully!', 'success')
    else:
        flash('Failed to save edict.', 'error')
    return redirect(url_for('admin_view_rules'))

@app.route('/admin/delete_rule/<rule_id>', methods=['POST'])
def admin_delete_rule(rule_id):
    edicts = load_edicts()
    edicts_to_keep = [rule for rule in edicts if rule.get('id') != rule_id]
    if len(edicts_to_keep) < len(edicts):
        if save_edicts(edicts_to_keep):
            flash(f'Edict {rule_id} deleted successfully!', 'success')
        else:
            flash(f'Failed to save changes after deleting {rule_id}.', 'error')
    else:
        flash(f'Edict {rule_id} not found.', 'warning')
    return redirect(url_for('admin_view_rules'))

@app.route('/admin/toggle_rule/<rule_id>', methods=['POST'])
def admin_toggle_rule(rule_id):
    edicts = load_edicts()
    rule_found = False
    for rule in edicts:
        if rule.get('id') == rule_id:
            rule['is_active'] = not rule.get('is_active', False)
            rule_found = True
            break
    if rule_found:
        if save_edicts(edicts):
            flash(f'Edict {rule_id} status toggled.', 'success')
        else:
            flash(f'Failed to save status toggle for {rule_id}.', 'error')
    else:
        flash(f'Edict {rule_id} not found for toggling.', 'warning')
    return redirect(url_for('admin_view_rules'))

# Add a simple root route for the admin UI
@app.route('/admin')
def admin_redirect():
    return redirect(url_for('admin_view_rules'))


if __name__ == '__main__':
    print("[BoTC Lexicon Admin] Awakening. Ensure 'botc_text_edicts.json' is present and writable.")
    print("Admin UI available at /admin/rules (once server is running)")
    # For development, it's good practice to ensure the templates folder and file exist
    # For simplicity, this is omitted here but assumed for render_template
    app.run(host='0.0.0.0', port=6661, debug=True)